local chunksize = 32

local function save_mod_settings()
  global.saved_settings = {}
  for _, setting in pairs(settings.startup) do
    if string.find(_, "alien%-biomes") then
      global.saved_settings[_] = setting
    end
  end
end

local function show_changed_settings()

  local text = "The current Alien Biomes mod settings for map generation do not match the settings that were used to create this map. \n\n"
  text = text .. "If you continue with the current settings and do not regenerate the terrain, then new terrain chunks will be created with the current settings and will not match the old terrain. \n\n"
  text = text .. "If you click Regenerate, then the existing terrain will be completely rebuilt to match the new terrain. (Some areas will be missing decoratives.) \n\n"
  text = text .. "This RESETS large sections of the map, you will probably lose some progress so don't overwrite your old save.\n\n"
  text = text .. "The regeneration process can take a long time. \n\n"
  text = text .. "To restore the original settings of this map, exit this map without saving, then in the Factorio main menu go to \n"
  text = text .. "Options > Mod Settings > Startup > Alien Biomes \n"
  text = text .. "and restore the original settings as shown below: "

  local settings_text = ""
  for _, setting in pairs(global.saved_settings) do
    if string.len(settings_text) > 0 then
       settings_text = settings_text .. ",\n"
    end
    local value = setting.value
    if value == true or value == false then
      value = value and "true" or "false"
    end
    settings_text = settings_text .. '"'.. _ .. '": { "value": ' .. value .. " }"
  end

  for _, player in pairs(game.connected_players) do
    player.gui.center.add{type = "frame", name="alien_biomes_settings_frame", style="alien-biomes-frame", direction="vertical", caption="Terrain Setting Mismatch"}
    player.gui.center.alien_biomes_settings_frame.add{type = "scroll-pane", name="alien_biomes_scroll", style="alien-biomes-scroll-pane", direction="vertical"}
    player.gui.center.alien_biomes_settings_frame.alien_biomes_scroll.add{
      type = "label",
      name="alien_biomes_settings_label",
      caption = text,
      style = "alien-biomes-label-multiline"
    }
    player.gui.center.alien_biomes_settings_frame.alien_biomes_scroll.add{
      type = "text-box",
      name="alien_biomes_settings_textbox",
      text = settings_text,
      style = "alien-biomes-textbox"
    }
    player.gui.center.alien_biomes_settings_frame.alien_biomes_scroll.add{
      type = "label",
      name="alien_biomes_settings_label2",
      caption = "If you know how to edit your mod-settings.json file you can  paste the above settings in the \"startup\" section.",
      style = "alien-biomes-label-multiline"
    }
    player.gui.center.alien_biomes_settings_frame.add{type = "flow", name="alien_biomes_buttons", direction="horizontal"}
    player.gui.center.alien_biomes_settings_frame.alien_biomes_buttons.add{type = "button", name="alien_biomes_settings_close", caption="Do Not Regenerate"}
    player.gui.center.alien_biomes_settings_frame.alien_biomes_buttons.add{type = "button", name="alien_biomes_settings_regenerate", caption="Regenerate Terrain"}
  end

end

local function offer_regenerate()

  local text = "Alien Biomes has been added to you map. If you continue and do not regenerate the terrain, then new terrain chunks generated will not match the existing terrain. \n\n"
  text = text .. "If you click Regenerate, then the existing terrain will be completely rebuilt to match the new terrain. (Some areas will be missing decoratives.) \n\n"
  text = text .. "This RESETS large sections of the map, you will probably lose some progress so don't overwrite your old save.\n\n"
  text = text .. "The regeneration process can take a long time."

  for _, player in pairs(game.connected_players) do
    player.gui.center.add{type = "frame", name="alien_biomes_settings_frame", style="alien-biomes-frame", direction="vertical", caption="Terrain Setting Mismatch"}
    player.gui.center.alien_biomes_settings_frame.add{type = "scroll-pane", name="alien_biomes_scroll", style="alien-biomes-scroll-pane", direction="vertical"}
    player.gui.center.alien_biomes_settings_frame.alien_biomes_scroll.add{
      type = "label",
      name="alien_biomes_settings_label",
      caption = text,
      style = "alien-biomes-label-multiline"
    }
    player.gui.center.alien_biomes_settings_frame.add{type = "flow", name="alien_biomes_buttons", direction="horizontal"}
    player.gui.center.alien_biomes_settings_frame.alien_biomes_buttons.add{type = "button", name="alien_biomes_settings_close", caption="Do Not Regenerate"}
    player.gui.center.alien_biomes_settings_frame.alien_biomes_buttons.add{type = "button", name="alien_biomes_settings_regenerate", caption="Regenerate Terrain"}
  end

end

local function assert_settings()

  if not global.saved_settings then return end

  for _, setting in pairs(settings.startup) do
    if string.find(_, "alien%-biomes") and global.saved_settings[_] then
      if setting.value ~= global.saved_settings[_].value then
        show_changed_settings()
        return
      end
    end
  end

end


local function chunk_to_position(chunk, for_tiles)
  return {chunk.x * chunksize, chunk.y * chunksize}
end

local function chunk_to_area(chunk, for_tiles)
  if for_tiles then
    return {{chunk.x * chunksize, chunk.y * chunksize}, {chunk.x * chunksize + chunksize -1, chunk.y * chunksize + chunksize -1}}
  else
    return {{chunk.x * chunksize, chunk.y * chunksize}, {chunk.x * chunksize + chunksize-0.0001, chunk.y * chunksize + chunksize-0.0001}}
  end
end

local function rebuild_nauvis()

  -- regenerate empty chunks
  -- for other chunks copy in tiles from nauvis-copy
  -- if new tile is water place water and old tile was not then place landfill if near a player entity

  local process_legacy_chunk_tiles = true
  local process_natural_entities = true
  local process_respawned_chunk_entities = true
  -- mark chunks as "legacy" or "respawned"
  log("rebuild_nauvis")

  local function log_chunk(chunk, message)
    --log("chunk "..chunk.x .." "..chunk.y.. " " .. message)
  end

  local nauvis_name = "nauvis"
  local nauvis_copy_name = "nauvis-copy"

  if not game.surfaces[nauvis_copy_name] then
    game.create_surface(nauvis_copy_name)
  end

  local nauvis = game.surfaces[nauvis_name]
  local nauvis_copy = game.surfaces[nauvis_copy_name]

  local chunks = {}

  for chunk in nauvis.get_chunks() do
    -- geting the chunk creates it
    local chunkdata = chunk
    table.insert(chunks, chunkdata)
    chunkdata.created = game.tick

    chunkdata.protected = false
    -- find force entities from player forces
    for _, force in pairs(game.forces) do
      if force.name ~= "neutral" and force.name ~= "enemy" then
        if nauvis.count_entities_filtered{
            area = chunk_to_area(chunk),
            force = force,
            limit = 1
          } > 0 then
            chunkdata.protected = true
            --log_chunk(chunk, "has entities of " .. force.name)
            break
        end
      end
    end

    -- find rail
    if not protected_chunk then
      if nauvis.count_entities_filtered{
          area = chunk_to_area(chunk),
          type = "straight-rail",
          limit = 1
        } > 0 then
          chunkdata.protected = true
          --log_chunk(chunk, "has rail")
      end
    end

    if chunkdata.protected then

      -- there's stuff here so keep the chunk and replace per-tile
      chunkdata.type = "legacy"
      nauvis_copy.request_to_generate_chunks({x = chunk.x * chunksize, y = chunk.y * chunksize}, 0)

    else

      --log_chunk(chunk, "can be respawned")
      -- delete the chunk then put any enemies back
      chunkdata.type = "respawned"
      chunkdata.pollution = nauvis.get_pollution({x = chunk.x + chunksize / 2, y = chunk.y + chunksize / 2})
      --if chunkdata.pollution > 0 then
        --log_chunk(chunk, "has pollution " .. chunkdata.pollution)
      --end

      local enemy_entities = nauvis.find_entities_filtered{
        area = chunk_to_area(chunk),
        force = "enemy"
      }
      if #enemy_entities > 0 then
        --log_chunk(chunk, "has " .. #enemy_entities .." enemies")
        chunkdata.enemy = {}
        for _, entity in pairs(enemy_entities) do
          if entity.type ~= "unit" then -- don't bother with units for now
            table.insert(chunkdata.enemy, { type = entity.type, name = entity.name, position = entity.position })
          end
        end
      end

      -- no player forces, just delete and remake
      nauvis.delete_chunk(chunk)

    end
  end

  nauvis_copy.force_generate_chunk_requests()


  local water_tiles = {
    ["water"] = true,
    ["deepwater"] = true,
    ["water-green"] = true,
    ["deepwater-green"] = true
  }

  local floor_tiles = {}
  for _, tile in pairs(game.tile_prototypes) do
    if tile.mineable_properties.minable then
      floor_tiles[tile.name] = true
    end
  end

  -- set legacy tiles
  if process_legacy_chunk_tiles then


    for _, chunk in pairs(chunks) do
      if chunk.type == "legacy" then
        local tiles = {}
        --log_chunk(chunk, "legacy processing")
        for y = 0, 31, 1 do
          for x = 0, 31, 1 do
            local position = {
              x = chunk.x * chunksize + x,
              y = chunk.y * chunksize + y
            }
            local old_tile_name = nauvis.get_tile(position.x, position.y).name
            if not floor_tiles[old_tile_name] then
                local new_tile_name = nauvis_copy.get_tile(position.x, position.y).name
                local new_is_water = water_tiles[new_tile_name]
                local old_is_water = water_tiles[old_tile_name]
                if new_is_water and not old_is_water then
                  local area = {{position.x, position.y}, {position.x+0.9999, position.y+0.9999}}
                  for _, force in pairs(game.forces) do
                    if force.name ~= "neutral" and force.name ~= "enemy" then
                      if nauvis.count_entities_filtered{
                          area = area,
                          force = force,
                          limit = 1
                        } > 0 then
                          new_tile_name = "landfill"
                          break
                      end
                    end
                  end
                  if nauvis.count_entities_filtered{
                      area = area,
                      type = "straight-rail",
                      limit = 1
                    } > 0 then
                      new_tile_name = "landfill"
                  end
                end
                table.insert(tiles, {name = new_tile_name, position = position})
            end
          end
        end
        local trees = nauvis.find_entities_filtered{type = "tree", area = chunk_to_area(chunk)}
        for _, tree in pairs(trees) do
          tree.destroy()
        end
        nauvis.set_tiles(tiles, true)
      end
    end
  end

  nauvis.regenerate_decorative() -- must be before deleted chunks generate
  --nauvis.regenerate_entity({"cliff", "small-cliff"})

  if process_natural_entities then
    local natural_entities = nauvis_copy.find_entities()
    --log("natural_entities: " .. #natural_entities)
    for _, natural in pairs(natural_entities) do
        local valid  = natural.type ~= "resource" and natural.type~="cliff"
        local area = natural.selection_box or natural.bounding_box
        --log("Copying decorative " .. natural.name)
        if valid and floor_tiles[nauvis.get_tile(natural.position.x, natural.position.y).name] then
          valid = false
        end
        if (valid and (natural.minable or natural.destructible or natural.type=="cliff") and area) then
         --log("Minable, selection box " .. serpent.block(area))
         area.left_top.x = area.left_top.x - 2
         area.left_top.y = area.left_top.y - 2
         area.right_bottom.x = area.right_bottom.x + 1.999
         area.right_bottom.y = area.right_bottom.y + 1.999
          if (nauvis.count_entities_filtered{
            area = area,
            limit = 1,
            force = "player"} > 0)
           or (nauvis.count_entities_filtered{
            area = area,
            limit = 1,
            force = "enemy"} > 0) then
                valid = false
          end
        end
        if valid then
            local options = {
            name=natural.name,
              position=natural.position,
              direction=natural.direction,
              force=natural.force,
              graphics_variation=natural.graphics_variation,
              bounding_box=natural.bounding_box,
              secondary_bounding_box=natural.secondary_bounding_box,
              selection_box=natural.selection_box,
              secondary_selection_box=natural.secondary_selection_box
            }
            if options.type == "cliff" then
                cliff_orientation=natural.cliff_orientation
            end
           if (not nauvis.create_entity(options)) then log("Can't create entity") end
        end
    end
  end

  game.delete_surface(nauvis_copy)

  for _, chunk in pairs(chunks) do
    -- warning: doing this during get chunks adds it to the current loop
    nauvis.request_to_generate_chunks({x = chunk.x * chunksize, y = chunk.y * chunksize}, 0)
  end

  nauvis.force_generate_chunk_requests()

  if process_respawned_chunk_entities then
    for _, chunk in pairs(chunks) do
      if chunk.type == "respawned" then

        --log_chunk(chunk, "respawned processing")
        if chunk.enemy then
          for _, entity in pairs(chunk.enemy) do
            if nauvis.can_place_entity(entity) then
              nauvis.create_entity(entity)
            end
          end
        end
        if chunk.pollution then
          nauvis.pollute({x = chunk.x + chunksize / 2, y = chunk.y + chunksize / 2}, chunk.pollution)
        end
      end
    end
  end

  game.print("Nauvis Regeneration Complete")
end

local function rebuild_nauvis_alt()

  local nauvis_name = "nauvis"
  local nauvis_copy_name = "nauvis-copy"
  local chunksize = 32

  if not game.surfaces[nauvis_copy_name] then
    game.create_surface(nauvis_copy_name)
  end

  local nauvis = game.surfaces[nauvis_name]
  local nauvis_copy = game.surfaces[nauvis_copy_name]

  local function chunk_key(x,y) return x*1000000000 + y end
  local chunks = {}
  for chunk in nauvis.get_chunks() do
    chunks[chunk_key(chunk.x,chunk.y)] = chunk
  end

  local deleted_chunks = { }
  -- make the new chunks
  for _, chunk in pairs(chunks) do
    local count = 0
    local area = {{chunk.x * chunksize - 10, chunk.y * chunksize - 10}, {chunk.x * chunksize + chunksize + 9, chunk.y * chunksize + chunksize + 9}}
    count = nauvis.count_entities_filtered{
        area = area,
        force = "player",
        limit = 1
      }

    if count == 0  then
        local enemies = nauvis.count_entities_filtered{
            area = area,
            force = "enemy",
            type="unit-spawner",
            limit = 1
          }
        if enemies > 0 and (
            nauvis.get_pollution(area[1]) > 0 or nauvis.get_pollution(area[2])>0
                or nauvis.count_entities_filtered{
                    area = {{area[1][1]-chunksize*3,area[1][2]-chunksize*3},{area[2][1]+chunksize*3,area[2][2]+chunksize*3}},
                    force = "player",
                    limit = 1  }>0) then
            count = enemies
        end
    end
    if count > 0 then
      -- there are forces here so copy tiles
      nauvis_copy.request_to_generate_chunks({x = chunk.x * chunksize, y = chunk.y * chunksize}, 0)
      nauvis.destroy_decoratives({{chunk.x * chunksize, chunk.y *chunksize}, {chunk.x * chunksize + chunksize - 1, chunk.y * chunksize + chunksize - 1}})
    else
      -- no forces, just delete and remake
      nauvis.delete_chunk(chunk)
      chunks[_] = nil
      deleted_chunks[_] = chunk
    end
  end

  nauvis_copy.force_generate_chunk_requests()

  -- copy tiles from generated copy to main
  local water_tiles = {["water-green"] = true,["deepwater-green"] = true,["water"] = true,["deepwater"] = true}
  local tiles = {}
  for _, chunk in pairs(chunks) do
    for y = 0, 31, 1 do
      for x = 0, 31, 1 do
        local position = {
          x = chunk.x * chunksize + x,
          y = chunk.y * chunksize + y
        }
        local tile_name = nauvis_copy.get_tile(position.x, position.y).name
        local copy = true
        if (water_tiles[tile_name]) then -- don't copy water over entities
         local count = 0
         count = nauvis.count_entities_filtered{
             area = {{x - math.random(7,9), y - math.random(7,9)}, {x + math.random(7,9), y + math.random(7,9)}},
             force = game.forces["player"],
             limit = 1
           }
         if count > 0 then
        --tile_name = "mineral-tan-dirt-1"
                  tile_name = nil
          end
        end
        if tile_name then
            table.insert(tiles, {name = tile_name, position = position})
            if #tiles > 20480 then
                nauvis.set_tiles(tiles, true)
                tiles = {}
            end
         end
      end
    end
  end
  if #tiles > 0 then nauvis.set_tiles(tiles, true) end

    local decoratives = nauvis_copy.find_entities()
    log("Found " .. #decoratives .. " decoratives")
    for _, d in pairs(decoratives) do
        local valid  = natural.type ~= "resource" and natural.type~="cliff"
        local area = natural.selection_box or natural.bounding_box
        --log("Copying decorative " .. natural.name)
        if (valid and (natural.minable or natural.destructible or natural.type=="cliff") and area) then
         --log("Minable, selection box " .. serpent.block(area))
         area.left_top.x = area.left_top.x - 2
         area.left_top.y = area.left_top.y - 2
         area.right_bottom.x = area.right_bottom.x + 2
         area.right_bottom.y = area.right_bottom.y + 2
          if (nauvis.count_entities_filtered{
            area = area,
            limit = 1,
            force = "player"} > 0)
           or (nauvis.count_entities_filtered{
            area = area,
            limit = 1,
            force = "enemy"} > 0) then
                valid = false
          end
        end
        if valid then
            local options = {
            name=natural.name,
            position=natural.position,
            direction=natural.direction,
            force=natural.force,
            graphics_variation=natural.graphics_variation,
            bounding_box=natural.bounding_box,
            secondary_bounding_box=natural.secondary_bounding_box,
            selection_box=natural.selection_box,
            secondary_selection_box=natural.secondary_selection_box
            }
            if options.type == "cliff" then
                cliff_orientation=natural.cliff_orientation
            end
           if (not nauvis.create_entity(options)) then log("Can't create entity: ".. natural.name) end
        end
    end
  --nauvis.regenerate_decorative({"cliff", "small-cliff"})
  -- nauvis.regenerate_decorative() -- does not work properly, everything is at highest density

  game.delete_surface(nauvis_copy)

  -- done, now smooth border tiles


  local step = 0 --doesn't work until next tick?
  local function continue(event)
    if step >= 10 then return end
    step = step + 1
    if step == 5 then
         --only chunks with living neighbours participiate
        for _, chunk in pairs(deleted_chunks) do
            if not chunks[chunk_key(chunk.x-1,chunk.y)]
             and not chunks[chunk_key(chunk.x+1,chunk.y)]
             and not chunks[chunk_key(chunk.x,chunk.y-1)]
             and not chunks[chunk_key(chunk.x,chunk.y+1)]
             and not chunks[chunk_key(chunk.x-1,chunk.y-1)]
             and not chunks[chunk_key(chunk.x+1,chunk.y+1)]
             then
               deleted_chunks[_] = nil
             else -- generate them so can edit
               nauvis.request_to_generate_chunks({x = chunk.x * chunksize+1, y = chunk.y * chunksize+1}, 0)
             end
        end

        nauvis.force_generate_chunk_requests()
        return
    end
    if step ~= 10 then return end

    -- copy border tiles from living neighbours over generated water tiles
    local tiles = {}
    for _, chunk in pairs(deleted_chunks) do

      log("Processing deleted chunk " .. chunk.x .. "," .. chunk.y ..", generated:" .. tostring(nauvis.is_chunk_generated({chunk.x,chunk.y})))
      for v = 1, 6, 1 do -- 6 neighbours
         local cx
         local cy
         local min = 0
              local max = 31
              if v == 1 then
                  cx = chunk.x
                  cy = chunk.y - 1
              elseif v == 2 then
                  cx = chunk.x
                  cy = chunk.y + 1
              elseif v == 3 then
                  cx = chunk.x - 1
                  cy = chunk.y
              elseif v == 4 then
                  cx = chunk.x + 1
                  cy = chunk.y
              elseif v == 5 then
                  cx = chunk.x - 1
                  cy = chunk.y - 1
                  min = 1
                  max = 1
              elseif v == 6 then
                  cx = chunk.x + 1
                  cy = chunk.y + 1
                  min = 1
                  max = 1
              end
          local chunkFromPos = {x = cx, y = cy}
          -- if there it's a living neightbour
          if (nauvis.is_chunk_generated(chunkFromPos)) and chunks[chunk_key(chunkFromPos.x, chunkFromPos.y)] then
              for m = min, max, 1 do
              local count = math.random(0,2)
              if count~=0 then
                  local position = { x = chunk.x * chunksize, y = chunk.y * chunksize }
                  local ox = 0
                  local oy = 0
                  if v == 1 then
                      position.x = position.x + m
                      position.y = position.y + 0
                      oy = -1
                  elseif v == 2 then
                      position.x = position.x + m
                      position.y = position.y + chunksize - 1
                      oy = 1
                  elseif v == 3 then
                      position.x = position.x + 0
                      position.y = position.y + m
                      ox = -1
                  elseif v == 4 then
                      position.x = position.x + chunksize - 1
                      position.y = position.y + m
                      ox = 1
                  elseif v == 5 then
                      position.x = position.x
                      position.y = position.y
                      ox = -1
                      oy = -1
                  elseif v == 6 then
                      position.x = position.x + chunksize - 1
                      position.y = position.y + chunksize - 1
                      ox = 1
                      oy = 1
                  end
                  local position_from = { x = position.x + ox, y = position.y + oy }
                  for c = 0,count-1,1 do
                      local dest = { x = position.x - ox * c, y = position.y - oy * c}
                      local current_name = nauvis.get_tile(dest.x, dest.y).name
                      if water_tiles[current_name] then
                         local new_name = nauvis.get_tile(position_from.x, position_from.y).name
                          if not water_tiles[new_name] then
                                 log("Copying tile from " .. serpent.block(position_from) .. " " .. new_name .." to " .. serpent.block(dest) .. " " .. current_name)
                               table.insert(tiles, {name = new_name, position = dest})
                          end
                      end
                  end
              end
            end
          end
      end
    end
    nauvis.set_tiles(tiles, true)
    game.print("Nauvis Regeneration Complete")
  end
  script.on_event(defines.events.on_tick, continue)
end

script.on_event(defines.events.on_gui_click, function(event)
  if event.element.name == "alien_biomes_settings_close" then
    game.players[event.player_index].gui.center.alien_biomes_settings_frame.destroy()
  elseif event.element.name == "alien_biomes_settings_regenerate" then
    game.players[event.player_index].gui.center.alien_biomes_settings_frame.destroy()
    rebuild_nauvis()
  end
end)


-- mod has changed or mod setting has changed
script.on_configuration_changed(function()

  assert_settings()

end)


script.on_event(defines.events.on_tick, function(event)
  if not global.saved_settings then

    save_mod_settings()
    if game.tick > 1 then
      -- alien biomes added to existing map,
      offer_regenerate()
    end
  end

end)
